const express = require('express');
const router = express.Router();
const emailService = require('../utils/emailService');
const { validateContactForm, handleValidationErrors } = require('../middleware/validation');

// Contact form submission
router.post('/', validateContactForm, handleValidationErrors, async (req, res) => {
  try {
    const { name, email, phone, enquiry } = req.body;

    // Log the submission (optional)
    console.log('Contact form submission received:', {
      name,
      email,
      phone: phone.replace(/\d(?=\d{4})/g, '*'), // Mask phone number in logs
      timestamp: new Date().toISOString()
    });

    // Send email
    const emailResult = await emailService.sendContactEmail({
      name,
      email,
      phone,
      enquiry
    });

    console.log('Contact email sent successfully:', emailResult.messageId);

    res.status(200).json({
      success: true,
      message: 'Contact form submitted successfully. We will get back to you soon!',
      data: {
        submissionId: emailResult.messageId,
        timestamp: new Date().toISOString()
      }
    });

  } catch (error) {
    console.error('Error processing contact form:', error);
    
    res.status(500).json({
      success: false,
      message: 'Failed to submit contact form. Please try again later.',
      error: process.env.NODE_ENV === 'development' ? error.message : undefined
    });
  }
});

// Health check for contact route
router.get('/health', (req, res) => {
  res.status(200).json({
    success: true,
    message: 'Contact service is running',
    timestamp: new Date().toISOString()
  });
});

module.exports = router;
