const request = require('supertest');
const app = require('./server');

// Simple test function (you can expand this with proper testing framework)
const testAPI = async () => {
  console.log('🚀 Starting API tests...\n');

  try {
    // Test health check
    console.log('1. Testing health check endpoint...');
    const healthResponse = await request(app).get('/');
    console.log('✅ Health check:', healthResponse.status === 200 ? 'PASSED' : 'FAILED');
    console.log('   Response:', healthResponse.body.message);

    // Test contact form validation
    console.log('\n2. Testing contact form validation...');
    const invalidContactResponse = await request(app)
      .post('/api/contact')
      .send({
        name: 'A', // Too short
        email: 'invalid-email',
        phone: '123',
        enquiry: 'Short' // Too short
      });
    
    console.log('✅ Contact validation:', invalidContactResponse.status === 400 ? 'PASSED' : 'FAILED');
    console.log('   Validation errors detected:', invalidContactResponse.body.errors?.length || 0);

    // Test enquiry form validation
    console.log('\n3. Testing enquiry form validation...');
    const invalidEnquiryResponse = await request(app)
      .post('/api/enquiry')
      .send({
        name: 'Test User',
        email: 'test@example.com',
        phone: '+1234567890',
        enquiry: 'Valid enquiry message with sufficient length to pass validation',
        category: '', // Empty category
        title: 'Hi' // Too short
      });
    
    console.log('✅ Enquiry validation:', invalidEnquiryResponse.status === 400 ? 'PASSED' : 'FAILED');
    console.log('   Validation errors detected:', invalidEnquiryResponse.body.errors?.length || 0);

    // Test enquiry categories endpoint
    console.log('\n4. Testing enquiry categories endpoint...');
    const categoriesResponse = await request(app).get('/api/enquiry/categories');
    console.log('✅ Categories endpoint:', categoriesResponse.status === 200 ? 'PASSED' : 'FAILED');
    console.log('   Categories available:', categoriesResponse.body.data?.length || 0);

    console.log('\n🎉 API tests completed successfully!');
    console.log('\n📝 Note: Email sending tests require proper Gmail configuration in .env file');
    
  } catch (error) {
    console.error('❌ Test failed:', error.message);
  }
};

// Run tests if this file is executed directly
if (require.main === module) {
  testAPI();
}

module.exports = testAPI;
